<?php
require_once '../php/includes/db.php';
require_once '../php/includes/functions.php';

header('Content-Type: application/json');

if (!isLoggedIn() || !isAdmin()) {
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

$action = $_POST['action'] ?? '';
$game_id = $_POST['game_id'] ?? 0;

if (!$game_id && $action !== 'generate_game') {
    echo json_encode(['error' => 'No game_id']);
    exit;
}

if ($action !== 'generate_game') {
    $stmt = $pdo->prepare("SELECT * FROM games WHERE id = ?");
    $stmt->execute([$game_id]);
    $game = $stmt->fetch();

    if (!$game) {
        echo json_encode(['error' => 'Game not found']);
        exit;
    }
    $drawn_numbers = $game['drawn_numbers'] ? json_decode($game['drawn_numbers'], true) : [];
}

try {
    $pdo->beginTransaction();

    if ($action === 'start_game') {
        $stmt = $pdo->prepare("UPDATE games SET status = 'active' WHERE id = ?");
        $stmt->execute([$game_id]);
        echo json_encode(['success' => true, 'message' => 'Juego iniciado']);
    } 
    elseif ($action === 'draw_ball') {
        if ($game['status'] !== 'active') {
            throw new Exception("El juego no está activo.");
        }

        // Generate Ball
        $available = array_diff(range(1, 75), $drawn_numbers);
        if (empty($available)) {
            throw new Exception("Ya se sortearon todos los números.");
        }

        $ball = $available[array_rand($available)];
        $drawn_numbers[] = $ball;
        $json_drawn = json_encode($drawn_numbers);

        $stmt = $pdo->prepare("UPDATE games SET drawn_numbers = ? WHERE id = ?");
        $stmt->execute([$json_drawn, $game_id]);

        // Check Winners
        // Get all cards for this game (sold and unsold/machine)
        $stmt = $pdo->prepare("SELECT * FROM cards WHERE game_id = ?");
        $stmt->execute([$game_id]);
        $cards = $stmt->fetchAll();

        $winner_found = false;
        $winner_ids = [];
        $is_machine_win = false;

        foreach ($cards as $card) {
            $card_nums = json_decode($card['numbers'], true);
            $flat_nums = [];
            $cols = ['B', 'I', 'N', 'G', 'O'];
            foreach ($cols as $col) {
                foreach ($card_nums[$col] as $idx => $n) {
                    if ($col == 'N' && $idx == 2) continue; // Skip free space (which is 0)
                    $flat_nums[] = $n;
                }
            }

            // Check if all flat_nums are in drawn_numbers
            $diff = array_diff($flat_nums, $drawn_numbers);
            if (empty($diff)) {
                // WINNER!
                $winner_found = true;
                $winner_ids[] = $card['user_id'];
                
                // Mark card as winner
                $u_stmt = $pdo->prepare("UPDATE cards SET is_winner = 1 WHERE id = ?");
                $u_stmt->execute([$card['id']]);
                
                // Only credit if it's a real user
                if ($card['user_id']) {
                    // Credit User
                    $u_stmt = $pdo->prepare("UPDATE users SET balance = balance + 100000 WHERE id = ?");
                    $u_stmt->execute([$card['user_id']]);
                    
                    // Log Transaction
                    $u_stmt = $pdo->prepare("INSERT INTO transactions (user_id, amount, type, description) VALUES (?, 100000, 'win', 'Premio Bingo Juego #$game_id')");
                    $u_stmt->execute([$card['user_id']]);
                } else {
                    $is_machine_win = true;
                }
            }
        }

        if ($winner_found) {
            // End Game
            $stmt = $pdo->prepare("UPDATE games SET status = 'finished', winner_user_id = ? WHERE id = ?");
            // If the first winner is NULL (machine), it will store NULL. 
            // If multiple winners (e.g. user and machine), we store the user if possible or just the first one.
            // Let's prioritize user if mixed, but for now just first one.
            $winner_id_to_store = $winner_ids[0];
            
            // If we have any human winner, store that ID. Otherwise NULL (Machine)
            foreach ($winner_ids as $wid) {
                if ($wid) {
                    $winner_id_to_store = $wid;
                    break;
                }
            }
            
            $stmt->execute([$winner_id_to_store, $game_id]);
            
            echo json_encode([
                'success' => true, 
                'ball' => $ball, 
                'winner' => true, 
                'winner_ids' => $winner_ids
            ]);
        } else {
            echo json_encode([
                'success' => true, 
                'ball' => $ball, 
                'winner' => false
            ]);
        }
    } 
    elseif ($action === 'end_game') {
        $stmt = $pdo->prepare("UPDATE games SET status = 'finished' WHERE id = ?");
        $stmt->execute([$game_id]);
        echo json_encode(['success' => true, 'message' => 'Juego finalizado']);
    }
    elseif ($action === 'generate_game') {
        // Logic to generate game and cards manually
        $date = date('Y-m-d');
        
        // Check if game exists
        $stmt = $pdo->prepare("SELECT id FROM games WHERE game_date = ?");
        $stmt->execute([$date]);
        $existing_game_id = $stmt->fetchColumn();
        
        if (!$existing_game_id) {
            $stmt = $pdo->prepare("INSERT INTO games (game_date, status) VALUES (?, 'scheduled')");
            $stmt->execute([$date]);
            $existing_game_id = $pdo->lastInsertId();
        }
        
        // Check card count
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM cards WHERE game_id = ?");
        $stmt->execute([$existing_game_id]);
        $count = $stmt->fetchColumn();
        
        if ($count < 200) {
            $needed = 200 - $count;
            $cards_data = [];
            for ($i = 0; $i < $needed; $i++) {
                $numbers = generateBingoCardNumbers();
                $cards_data[] = "($existing_game_id, '$numbers', 3000.00)";
            }
            
            // Batch insert in chunks
            $chunks = array_chunk($cards_data, 50);
            foreach ($chunks as $chunk) {
                $sql = "INSERT INTO cards (game_id, numbers, price) VALUES " . implode(',', $chunk);
                $pdo->query($sql);
            }
            echo json_encode(['success' => true, 'message' => "Juego creado y $needed cartones generados para hoy."]);
        } else {
            echo json_encode(['success' => true, 'message' => "El juego ya existe y tiene suficientes cartones."]);
        }
    }

    $pdo->commit();

} catch (Exception $e) {
    $pdo->rollBack();
    echo json_encode(['error' => $e->getMessage()]);
}
?>
