document.addEventListener('DOMContentLoaded', function() {
    if (typeof GAME_ID === 'undefined' || GAME_ID === null) return;

    // Elements
    const ballHistory = document.getElementById('ball-history');
    const currentBallSpotlight = document.getElementById('current-ball');
    const currentBallNum = document.getElementById('current-ball-num');
    const statusDisplay = document.getElementById('game-status-badge');
    
    // Chat Elements
    const chatMessages = document.getElementById('chat-messages');
    const chatInput = document.getElementById('chat-input');
    const btnSendChat = document.getElementById('btn-send-chat');
    const chatSidebar = document.getElementById('chat-sidebar');
    const toggleChatBtn = document.getElementById('toggle-chat-btn');
    const mobileChatFab = document.getElementById('mobile-chat-fab'); // New FAB
    const closeChatBtn = document.getElementById('close-chat');

    let knownDrawnNumbers = [];
    let lastChatId = 0;

    // --- Helpers ---
    function getBallClass(number) {
        number = parseInt(number);
        if (number <= 15) return 'ball-b';
        if (number <= 30) return 'ball-i';
        if (number <= 45) return 'ball-n';
        if (number <= 60) return 'ball-g';
        return 'ball-o';
    }

    function getBallLetter(number) {
        number = parseInt(number);
        if (number <= 15) return 'B';
        if (number <= 30) return 'I';
        if (number <= 45) return 'N';
        if (number <= 60) return 'G';
        return 'O';
    }

    // --- Toggle Chat Logic ---
    function toggleChat() {
        if (chatSidebar) {
            chatSidebar.classList.toggle('active');
            chatSidebar.classList.toggle('collapsed'); // Maintain compatibility or remove if not needed
        }
    }

    if (toggleChatBtn) toggleChatBtn.addEventListener('click', toggleChat);
    if (mobileChatFab) mobileChatFab.addEventListener('click', toggleChat);
    if (closeChatBtn) closeChatBtn.addEventListener('click', () => {
        chatSidebar.classList.remove('active');
        chatSidebar.classList.add('collapsed');
    });

    // --- Game Logic ---
    function updateGame() {
        fetch(`/api/get_state.php?game_id=${GAME_ID}`)
            .then(response => response.json())
            .then(data => {
                if (data.error) {
                    console.error(data.error);
                    return;
                }

                // Update Status
                if (statusDisplay) {
                    statusDisplay.textContent = data.status;
                    if (data.status === 'active') {
                        statusDisplay.style.background = '#28a745'; // Green
                    } else if (data.status === 'finished') {
                        statusDisplay.style.background = '#6c757d'; // Gray
                    }
                }

                // Handle Finish
                if (data.status === 'finished') {
                    const winModal = document.getElementById('win-modal');
                    if (winModal && winModal.style.display === 'none') {
                        if (data.winner_name) {
                            const winText = winModal.querySelector('.win-modal-text');
                            if (winText) winText.textContent = `¡El ganador es ${data.winner_name}!`;
                        }
                        winModal.style.display = 'flex';
                        // Trigger confetti
                        if (typeof confetti !== 'undefined') {
                            confetti({
                                particleCount: 150,
                                spread: 70,
                                origin: { y: 0.6 }
                            });
                        }
                    }
                }

                // Update Balls
                const newNumbers = data.drawn_numbers.filter(n => !knownDrawnNumbers.includes(n));
                
                if (newNumbers.length > 0) {
                    // Clear placeholder if first ball
                    if (knownDrawnNumbers.length === 0 && ballHistory) {
                        ballHistory.innerHTML = '';
                    }

                    newNumbers.forEach(n => {
                        knownDrawnNumbers.push(n);
                        
                        const ballClass = getBallClass(n);
                        const letter = getBallLetter(n);
                        
                        // Update Spotlight
                        if (currentBallSpotlight && currentBallNum) {
                            currentBallSpotlight.className = `ball-spotlight ${ballClass}`; // Add color class to spotlight
                            // Keep base class
                            currentBallSpotlight.classList.add('ball-spotlight'); 
                            
                            currentBallNum.textContent = n;
                            const letterSpan = currentBallSpotlight.querySelector('.ball-letter');
                            if (letterSpan) letterSpan.textContent = letter;

                            // Animation reset
                            currentBallSpotlight.style.animation = 'none';
                            currentBallSpotlight.offsetHeight; /* trigger reflow */
                            currentBallSpotlight.style.animation = 'bounceIn 0.8s';
                        }

                        // Add to History
                        if (ballHistory) {
                            const ball = document.createElement('div');
                            ball.className = `ball-3d ${ballClass} new`;
                            ball.innerHTML = `
                                <span class="ball-letter">${letter}</span>
                                <span>${n}</span>
                            `;
                            // Prepend to show newest first? Or append? 
                            // Usually history moves left, so append.
                            ballHistory.appendChild(ball);
                            // Scroll to end
                            ballHistory.scrollLeft = ballHistory.scrollWidth;
                        }
                        
                        // Check cards
                        const cellId = `cell-${n}`; // This needs to be specific to user cards
                        // Actually we need to check all cells with this number
                        // The PHP generates IDs like `cell-{cardId}-{number}`
                        // We need to select all elements ending with -{number}
                        const matchingCells = document.querySelectorAll(`[id$="-${n}"]`);
                        matchingCells.forEach(cell => {
                            cell.classList.add('marked');
                            // Add animation
                            cell.style.animation = 'pulse 0.5s';
                        });
                    });
                }
            })
            .catch(err => console.error(err));
    }

    function markCards(number) {
        // Find all cells with this number
        const selector = `[id$="-${number}"]`;
        const cells = document.querySelectorAll(selector);
        
        cells.forEach(cell => {
            if (!cell.classList.contains('marked')) {
                cell.classList.add('marked');
                // Trigger visual feedback if needed
            }
        });
    }

    // --- Chat Logic ---
    function updateChat() {
        if (!chatMessages) return;

        fetch(`/api/chat.php?game_id=${GAME_ID}&last_id=${lastChatId}`)
            .then(response => response.json())
            .then(data => {
                if (data.messages && data.messages.length > 0) {
                    data.messages.forEach(msg => {
                        const div = document.createElement('div');
                        div.style.marginBottom = '8px';
                        div.style.padding = '8px';
                        div.style.background = 'white';
                        div.style.borderRadius = '8px';
                        div.style.boxShadow = '0 2px 5px rgba(0,0,0,0.05)';
                        div.style.fontSize = '0.9rem';
                        
                        div.innerHTML = `<strong style="color: var(--primary);">${msg.user_name}:</strong> ${msg.message}`;
                        chatMessages.appendChild(div);
                        lastChatId = msg.id;
                    });
                    chatMessages.scrollTop = chatMessages.scrollHeight;
                }
            })
            .catch(err => console.error(err));
    }

    function sendChat() {
        if (!chatInput) return;
        const msg = chatInput.value.trim();
        if (!msg) return;

        const formData = new FormData();
        formData.append('game_id', GAME_ID);
        formData.append('message', msg);

        fetch('/api/chat.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                chatInput.value = '';
                updateChat();
            }
        });
    }

    // --- Event Listeners ---
    if (btnSendChat) {
        btnSendChat.addEventListener('click', sendChat);
    }
    if (chatInput) {
        chatInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') sendChat();
        });
    }

    // Sidebar Toggle
    if (toggleChatBtn && chatSidebar) {
        toggleChatBtn.addEventListener('click', () => {
            if (window.innerWidth <= 992) {
                chatSidebar.classList.toggle('active');
            } else {
                chatSidebar.classList.toggle('collapsed');
            }
        });
    }
    
    if (closeChatBtn && chatSidebar) {
        closeChatBtn.addEventListener('click', () => {
            if (window.innerWidth <= 992) {
                chatSidebar.classList.remove('active');
            } else {
                chatSidebar.classList.add('collapsed');
            }
        });
    }

    // Initialize
    updateGame();
    updateChat();
    setInterval(updateGame, 3000);
    setInterval(updateChat, 3000);
});
